<?php

namespace app\common\model;

use think\facade\Log;
use think\Model;

class Order extends Model
{
    protected $autoWriteTimestamp = true;
    protected $createTime = 'create_time';
    protected $updateTime = false;
    
    public function goods()
    {
        return $this->hasOne(Goods::class,'id','goods_id');
    }

    /**
     * @param int $uid
     * @param array $extra_data 额外参数
     * @return array
     * @throws \Exception
     */
public function createNewOrder($uid, $extra_data) 
{
    $user_info = User::alias('u')
        ->leftJoin('mod_level l', 'u.level_id=l.id')
        ->where(['u.id' => $uid])
        ->field('u.credit_score,u.status,u.balance,u.trade_status,u.order_total,u.deal_date,u.today_profit,l.order_num,l.order_rate,l.limit_price')
        ->find();

    if ($user_info['credit_score'] < 100) throw new \Exception('信用分低于100');
    if ($user_info['status'] != 1) throw new \Exception('该账号被禁用了，请联系客服');
    if ($user_info['trade_status'] != 1) throw new \Exception('交易被禁用，请联系客服');
    if ($user_info['balance'] < 1) throw new \Exception('余额不足，请充值-2-' . $user_info['limit_price']);
    if ($user_info['limit_price'] > $user_info['balance']) throw new \Exception('会员等级余额不足-0-' . $user_info['limit_price']);
    if ($user_info['order_total'] >= $user_info['order_num']) throw new \Exception('可做单数已用完-1-' . $user_info['order_total']);

    $rule_info = Schedule::where(['uid' => $uid, 'status' => 1])->find();
    $goods_info = [];
    $order_no = $user_info['order_total'] + 1;

    if (!empty($rule_info)) {
        $rule = json_decode($rule_info['rule'], true) ?? [];
        if (!empty($rule[$order_no])) {
            $goods_info = Goods::where(['id' => $rule[$order_no]['gid']])->field('id,price,name,img')->find();
        }
    }

    if (!$goods_info) {
        $goods_info = Goods::where('price', '<', $user_info['balance'])
            ->field('id,price,name,img')
            ->orderRaw('rand()')
            ->find();

        if (!$goods_info) throw new \Exception('符合商品不存在，请联系客服');
    }

    if ($user_info['deal_date'] != date('Ymd')) {
        $user_data['deal_date'] = date('Ymd');
        $user_data['today_profit'] = 0;
    }

    $user_data['order_total'] = $order_no;

    // ✅ Ensure order_rate is retrieved correctly and converted to a float
    $order_rate = floatval($user_info['order_rate']);
    if (!empty($rule_info)) {
        $rule = json_decode($rule_info['rule'], true) ?? [];
        if (!empty($rule[$order_no]) && isset($rule[$order_no]['order_rate'])) {
            $order_rate = floatval($rule[$order_no]['order_rate']);
        }
    }

    // ✅ Fix: Negative order_rate is properly handled
    $comm = round($goods_info['price'] * $order_rate, 2);

    $order = [
        'uid' => $uid,
        'order_no' => getSn('SC'),
        'goods_id' => $goods_info['id'],
        'num' => 1,
        'price' => $goods_info['price'],
        'commission' => $comm,
        'create_time' => time(),
    ];

    $log = [
        'uid' => $uid,
        'type' => 3,
        'status' => 2,
        'price' => $goods_info['price'],
        'price_pre' => $user_info['balance'],
        'explain' => '生成订单扣款',
        'extra_id' => $goods_info['id'],
    ];

    $info = [
        'goods_name' => $goods_info['name'],
        'price' => $goods_info['price'],
        'goods_img' => $goods_info['img'],
        'num' => 1,
        'order_no' => $order['order_no'],
        'commission' => $comm,
        'create_time' => $order['create_time'],
    ];

    $this->startTrans();
    try {
        $log['order_id'] = $info['order_id'] = $this->insertGetId($order);

        // ✅ Update user balance first
        $userUpdate = User::where(['id' => $uid])
            ->inc('freeze_balance', $goods_info['price'])
            ->dec('balance', $goods_info['price']);

        // ✅ Ensure correct balance adjustments for negative commissions
        if ($comm > 0) {
            // Positive commission (user earns money)
            $userUpdate->inc('commission', $comm)->inc('today_profit', $comm);
        } elseif ($comm < 0) {
            // Negative commission (user loses money)
            $comm_abs = abs($comm);
            $userUpdate->dec('commission', $comm_abs)->dec('balance', $comm_abs);

            // ✅ Fix: Accumulate today's profit instead of replacing it
            User::where(['id' => $uid])->inc('today_profit', -$comm_abs)->update();
        }

        // ✅ Ensure today_profit is updated in all cases
        $userUpdate->update($user_data);

        // ✅ Log transaction for debugging
        $todayProfitBefore = $user_info['today_profit'];
        $todayProfitAfter = $todayProfitBefore - abs($comm);
        Log::info("User: $uid | Commission: $comm | Today Profit Before Update: $todayProfitBefore");
        Log::info("User: $uid | Today Profit After Update: $todayProfitAfter");

        WalletLog::create($log);
        $this->commit();
    } catch (\Exception $e) {
        $this->rollback();
        $info = [];
        Log::error('购买订单-' . $e->getMessage() . ':' . $e->getLine());
    }

    return $info;
}





    /**
     * @param int $order_id
     * @param int $uid
     * @param array $level_arr 分佣比例
     * @throws \Exception
     */
    public function doOrder($order_id, $uid, $level_arr)
    {
        $order_info = $this->find($order_id);
        if (!$order_info)
            throw new \Exception('数据不存在');
        if ($order_info['uid'] != $uid)
            throw new \Exception('参数错误，请确认订单号!');
        if ($order_info['status'] != 1)
            throw new \Exception('订单已处理过，请刷新');
        //获取上级族谱
        $user_info = User::field('family,balance')->find($uid);
        if ($user_info['balance'] < 0)
            throw new \Exception('可用余额不足');
        $this->startTrans();
        try {
            $add_price = $order_info['price'] + $order_info['commission'];
            //返佣
            User::where(['id' => $uid])->inc('balance', $add_price)->dec('freeze_balance', $order_info['price'])
                ->inc('commission', $order_info['commission'])->inc('today_profit', $order_info['commission'])->update();
            //返佣记录
            $log[] = [
                'uid' => $uid,
                'order_id' => $order_info['id'],
                'type' => 8,
                'status' => 1,
                'price' => $order_info['price'],
                'price_pre' => $user_info['balance'],
                'explain' => '交易金额返回',
                'create_time' => time(),
                'extra_id' => $order_info['goods_id'],
            ];
            $log[] = [
                'uid' => $uid,
                'order_id' => $order_info['id'],
                'type' => 4,
                'status' => 1,
                'price' => $order_info['commission'],
                'price_pre' => $user_info['balance'] + $order_info['price'],
                'explain' => '交易返佣',
                'create_time' => time(),
                'extra_id' => $order_info['goods_id'],
            ];
            $log[] = [
                'uid' => $uid,
                'order_id' => $order_info['id'],
                'type' => 9,
                'status' => 1,
                'price' => $user_info['balance'] + $add_price,
                'price_pre' => $user_info['balance'] + $add_price,
                'explain' => '交易完成后余额',
                'create_time' => time(),
                'extra_id' => $order_info['goods_id'],
            ];
            if (!empty($user_info['family'])) {
                $family = explode('-', $user_info['family']);
                $child_id = $uid;
                foreach ($family as $level => $pid) {
                    $parent = User::field('balance')->find($pid);
                    $comm = round($order_info['commission'] * $level_arr[$level], 2);
if ($comm != 0) {
    $log[] = [
        'uid' => $pid,
        'order_id' => $order_info['id'],
        'type' => 5,
        'status' => 1,
        'price' => $comm,
        'price_pre' => $parent['balance'],
        'explain' => ($comm > 0) ? ($level + 1) . '级返佣金' : '无效佣金',
        'create_time' => time(),
        'extra_id' => $child_id,
    ];
    User::where(['id' => $pid])->inc('balance', $comm)->inc('commission', $comm)->update();
}
                    $child_id = $pid;
                }
            }
            $this->where(['id' => $order_id])->update(['status' => 2, 'com_status' => 2, 'end_time' => time()]);
            WalletLog::insertAll($log);
            $this->commit();
        } catch (\Exception $e) {
            $this->rollback();
            Log::error('做单失败-' . $e->getMessage() . ':' . $e->getLine());
        }

    }

}
